#include "main.h"

#include <string>
#include <vector>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <Shlwapi.h>

#include "macro.h"
#include "luaglue.h"
#include "luaengine.h"
#include "misc.h"
#include "color.h"
#include "filesystem.h"
#include "logger.h"

#include "error.h"

#include "settings.h"

int mainRunning;

int restoreDefaults();
void splitArgs(std::string cmd, std::vector<std::string> &args);
std::string strReplaceAll(std::string, std::string, std::string);
std::string promptForScript(std::vector<std::string> &args);

static BOOL WINAPI consoleControlCallback(DWORD dwCtrlType);

int main(int argc, char *argv[])
//int WINAPI WinMain(HINSTANCE hThisInstance, HINSTANCE hPrevInstance,
//	LPSTR lpszArgument, int nCmdShow)
{
	//int argc = __argc;
	//char **argv = __argv;
	mainRunning = true;
	SetConsoleCtrlHandler(consoleControlCallback, true);

	CLI_OPTIONS clio;
	handleCommandLineArguments(argc, argv, &clio);
	SetCurrentDirectory( getAppPath().c_str() );

	#ifdef DISPLAY_DEBUG_MESSAGES
		printf("Test?\n");
		Logger::instance()->add("DEBUG: Initializing LuaEngine.\n");
	#endif

	LuaEngine engine;
	if( !engine.init(&clio) ) {
		fprintf(stderr, "Error initializing Lua Engine\n");
		exit(1); }

	#ifdef DISPLAY_DEBUG_MESSAGES
		Logger::instance()->add("DEBUG: Entering main loop.");
	#endif

	bool autoload = (clio.scriptSet);
	while(mainRunning)
	{
		restoreDefaults();

		if( !autoload) // Grab filename &args from user
		{
			clio.args.clear();
			std::string filename = promptForScript(clio.args);
			clio.scriptName = filename;
		}

		if( clio.scriptName == "" )
		{
			printf("\nYou did not enter a script to run. Let\'s try that again.\n\n");
			continue;
		}

		if( !engine.checkCommands(clio.scriptName, clio.args) )
		{ // Not a command, but a script to run. So run it.
			if( !autoload )
			{
				// Fix script path (if needed)
				if( PathIsRelative(clio.scriptName.c_str()) )
				{
					std::string replacescript;
					std::string scriptsDir = Settings::instance()->getString(CONFIG_VAR_SCRIPTS_DIR);
					if( scriptsDir.length() > 0 )
						replacescript = scriptsDir;
					else
						replacescript = DEFAULT_SCRIPTS_DIR;
					replacescript += clio.scriptName;
					clio.scriptName = replacescript;

					// Set execution path
					std::string exePath = getAppPath() + "/";
					exePath += getFilePath(clio.scriptName);
					Macro::instance()->setExecutionPath(exePath);
				}
				else
				{
					// Set execution path
					Macro::instance()->setExecutionPath(getFilePath(clio.scriptName));
				}

			}
			else
			{
				// Set execution path
				Macro::instance()->setExecutionPath(
					getFilePath(clio.scriptName));
			}

			engine.run(clio.scriptName, clio.args);
		}

		// Disable autoloading now that it has been used (if needed)
		autoload = false;

		if( mainRunning )
		{
			#ifdef	DISPLAY_DEBUG_MESSAGES
			Logger::instance()->add("DEBUG: Reinitializing engine.");
			#endif
			if( !engine.reinit(&clio) )
			{
				const char *err = "Failed to re-initialize Lua Engine.\n";
				fprintf(stderr, err);
				Logger::instance()->add(err);
			}
			#ifdef	DISPLAY_DEBUG_MESSAGES
			Logger::instance()->add("DEBUG: Reinit OK.");
			#endif
		}
	}
	#ifdef	DISPLAY_DEBUG_MESSAGES
	Logger::instance()->add("DEBUG: Engine cleanup.");
	#endif
	engine.cleanup();
	return 0;
}

int restoreDefaults()
{
	setColor(COLOR_LIGHTGRAY);
	return 0;
}


void splitArgs(std::string cmd, std::vector<std::string> &args)
{
	unsigned int startpos = 0;
	unsigned int lastpos = 0;

	// Take quotes, replace sub-spaces
	startpos = cmd.find("\"");
	while( startpos != std::string::npos )
	{
		// Find the next quote
		unsigned int nextpos = cmd.find("\"", startpos+1);
		if( nextpos - startpos > 0 )
		{
			// Grab substring, remove quotes, substitute spaces
			std::string substr = cmd.substr(startpos+1, nextpos - startpos - 1);
			substr = strReplaceAll(substr, " ", "$_SPACE_$");
			// Now pop it back into place.
			cmd.replace(startpos, nextpos - startpos + 1, substr);
			lastpos = nextpos+1;
			startpos = cmd.find("\"", lastpos);
		}
		else
			break;
	}

	// Now split by spaces
	startpos = cmd.find(' ');
	args.push_back( strReplaceAll(cmd.substr(0, startpos), "$_SPACE_$", " ") );

	lastpos = startpos;
	while(startpos != std::string::npos)
	{
		startpos = cmd.substr(lastpos+1).find(' ');
		std::string tmp = cmd.substr(lastpos+1, startpos);
		if( tmp != "" )
		{
			args.push_back( strReplaceAll(tmp, "$_SPACE_$", " ") );
		}
		lastpos += startpos +1;
	}
}

std::string strReplaceAll(std::string instr, std::string search, std::string replace)
{
	// First we need to handle subquotes
	unsigned int i = instr.find(search);
	while( i != std::string::npos )
	{
		instr.replace(i, search.length(), replace);
		i = instr.find(search, i+replace.length());
	}

	return instr;
}

std::string promptForScript(std::vector<std::string> &args)
{
	static std::string previousScript = "";
	std::string filename;

	while( kbhit() ) getch(); // Clear keyboard buffer

	printf("Please enter the script name to run.\n");
	printf("Type in \'exit\' (without quotes) to exit.\n");
	printf("Script> ");

	std::string fullcmd;
	std::cin.clear();
	getline(std::cin, fullcmd);
	std::cin.clear();

	if( fullcmd == "" )
	{ // No input, show file open dialog
        std::string filter = "Lua files";
        filter.push_back('\0');
        filter.append("*.lua");
        filter.push_back('\0');
        filter.append("All files");
        filter.push_back('\0');
		filter.append("*.*");
		filter.push_back('\0');
		filter.push_back('\0');

		std::string fullScriptsDir = "";
		if( previousScript.length() > 0 )
		{
			fullcmd = Macro::instance()->getOpenFileName(previousScript, filter);
		}
		else
		{
			std::string scriptsDir = Settings::instance()->getString(CONFIG_VAR_SCRIPTS_DIR);
			if( scriptsDir.length() == 0 )
				scriptsDir = DEFAULT_SCRIPTS_DIR;

			fullScriptsDir = getAppPath() + "/";
			fullScriptsDir += scriptsDir;
			fullcmd = Macro::instance()->getOpenFileName(fullScriptsDir, filter);
		}
		previousScript = fullcmd;
		fullcmd = std::string("\"") + fullcmd + std::string("\"");
	}

	splitArgs(fullcmd, args);

	fflush(stdin);
	while( kbhit() ) getch(); // Clear keyboard buffer

	filename = args[0];
	args.erase(args.begin()); // Don't return filename in args array
	return filename;
}

static BOOL WINAPI consoleControlCallback(DWORD dwCtrlType)
{
	switch(dwCtrlType)
	{
		case CTRL_C_EVENT:
		case CTRL_CLOSE_EVENT:
		case CTRL_LOGOFF_EVENT:
		case CTRL_SHUTDOWN_EVENT:
			/* We pretty much have to call exit(0) here.
				Would be nice to be able to just set mainRunning = false
				and let it exit graciously, but, that's WIN32 for you.
			*/
			mainRunning = false;
			exit(0);
			return true;
		break;
	}
	return false;
}
